/*
   This file is part of TALER
   Copyright (C) 2024 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_token_family_key.c
 * @brief Implementation of the lookup_token_family_key function for Postgres
 * @author Christian Blättler
 */
#include "platform.h"
#include <gnunet/gnunet_pq_lib.h>
#include <gnunet/gnunet_time_lib.h>
#include <string.h>
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_token_family_key.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_token_family_key (
  void *cls,
  const char *instance_id,
  const char *token_family_slug,
  struct GNUNET_TIME_Timestamp min_valid_after,
  struct GNUNET_TIME_Timestamp max_valid_after,
  struct TALER_MERCHANTDB_TokenFamilyKeyDetails *details)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_string (token_family_slug),
    GNUNET_PQ_query_param_timestamp (&min_valid_after),
    GNUNET_PQ_query_param_timestamp (&max_valid_after),
    GNUNET_PQ_query_param_end
  };

  check_connection (pg);
  PREPARE (pg,
           "lookup_token_family_key",
           "SELECT"
           " h_pub"
           ",pub"
           ",priv"
           ",cipher"
           ",merchant_token_family_keys.valid_after as key_valid_after"
           ",merchant_token_family_keys.valid_before as key_valid_before"
           ",slug"
           ",name"
           ",description"
           ",description_i18n"
           ",merchant_token_families.valid_after"
           ",merchant_token_families.valid_before"
           ",duration"
           ",kind"
           ",issued"
           ",used"
           " FROM merchant_token_families"
           " LEFT JOIN merchant_token_family_keys"
           "   ON merchant_token_families.token_family_serial = merchant_token_family_keys.token_family_serial"
           "   AND merchant_token_family_keys.valid_after >= $3"
           "   AND merchant_token_family_keys.valid_after <= $4"  /* TODO: Should this be < instead of <=? */
           " JOIN merchant_instances"
           "   USING (merchant_serial)"
           " WHERE merchant_instances.merchant_id=$1"
           "   AND slug=$2"
           " ORDER BY merchant_token_family_keys.valid_after ASC"
           " LIMIT 1");

  if (NULL == details)
  {
    struct GNUNET_PQ_ResultSpec rs_null[] = {
      GNUNET_PQ_result_spec_end
    };

    check_connection (pg);
    return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                     "lookup_token_family_key",
                                                     params,
                                                     rs_null);
  }

  {
    char *kind;
    enum GNUNET_DB_QueryStatus qs;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_blind_sign_pub ("pub",
                                              &details->pub.public_key),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_blind_sign_priv ("priv",
                                               &details->priv.private_key),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_timestamp ("key_valid_after",
                                         &details->valid_after),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_timestamp ("key_valid_before",
                                         &details->valid_before),
        NULL),
      GNUNET_PQ_result_spec_string ("slug",
                                    &details->token_family.slug),
      GNUNET_PQ_result_spec_string ("name",
                                    &details->token_family.name),
      GNUNET_PQ_result_spec_string ("description",
                                    &details->token_family.description),
      TALER_PQ_result_spec_json ("description_i18n",
                                 &details->token_family.description_i18n),
      GNUNET_PQ_result_spec_timestamp ("valid_after",
                                       &details->token_family.valid_after),
      GNUNET_PQ_result_spec_timestamp ("valid_before",
                                       &details->token_family.valid_before),
      GNUNET_PQ_result_spec_relative_time ("duration",
                                           &details->token_family.duration),
      GNUNET_PQ_result_spec_string ("kind",
                                    &kind),
      GNUNET_PQ_result_spec_uint64 ("issued",
                                    &details->token_family.issued),
      GNUNET_PQ_result_spec_uint64 ("used",
                                    &details->token_family.used),
      GNUNET_PQ_result_spec_end
    };

    memset (details,
            0,
            sizeof (*details));
    qs = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "lookup_token_family_key",
                                                   params,
                                                   rs);
    if (GNUNET_DB_STATUS_SUCCESS_ONE_RESULT == qs)
    {
      if (0 == strcmp (kind,
                       "discount"))
        details->token_family.kind = TALER_MERCHANTDB_TFK_Discount;
      else if (0 == strcmp (kind,
                            "subscription"))
        details->token_family.kind = TALER_MERCHANTDB_TFK_Subscription;
      else
      {
        GNUNET_free (kind);
        GNUNET_break (0);
        return GNUNET_DB_STATUS_HARD_ERROR;
      }
      GNUNET_free (kind);
    }
    return qs;
  }
}
