#!/usr/bin/env node

// Bump the package.json versions in Taler-related packagesin
// this repository.
// The version must be in one of the following formats:
// - x.y.z
// - x.y.z-dev.n

import * as child_process from "child_process";
import * as fs from "fs";

let requestedVersion = process.argv[2];

let dry = false;
if (process.argv.includes("--dry")) {
  dry = true;
}

let verMatched = false;
let verMajor = 0;
let verMinor = 0;
let verPatch = 0;
let verDev = undefined;

// Parse the requested version
const releaseVerRegex = /^(\d+)[.](\d+)[.](\d+)$/;
const devVerRegex = /^(\d+)[.](\d+)[.](\d+)-dev[.](\d+)$/;

const releaseMatch = requestedVersion.match(releaseVerRegex);
if (releaseMatch) {
  verMatched = true;
  verMajor = releaseMatch[1];
  verMinor = releaseMatch[2];
  verPatch = releaseMatch[3];
}

if (!verMatched) {
  const devMatch = requestedVersion.match(devVerRegex);
  if (devMatch) {
    verMatched = true;
    verMajor = devMatch[1];
    verMinor = devMatch[2];
    verPatch = devMatch[3];
    verDev = devMatch[4];
  }
}

if (!verMatched) {
  console.error('wrong version format, it should have 3 numbers separated with dots or end with -dev.<buildnumber>')
  process.exit(1)
}

const packages = fs.readdirSync("packages")

for (const pkg of packages) {
  const p = `packages/${pkg}/package.json`;
  const data = JSON.parse(fs.readFileSync(p));
  console.log(p, data.version);
  if (!dry) {
    data.version = requestedVersion;
    fs.writeFileSync(p, JSON.stringify(data, undefined, 2) + "\n");
  }
}

{
  const p = "packages/taler-wallet-webextension/manifest-common.json";
  const data = JSON.parse(fs.readFileSync(p));
  console.log("manifest version", data.version);
  console.log("manifest version_name", data.version_name);
  // In manifest.json, we transform x.y.z-dev.n info x.y.z.n.
  // It's necessary because browsers only allow decimals and dots
  // in that field.
  let dottedVer = undefined;
  if (verDev != null) {
    dottedVer = `${verMajor}.${verMinor}.${verPatch}.${verDev}`;
  } else {
    dottedVer = `${verMajor}.${verMinor}.${verPatch}`;
  }
  console.log("new manifest version", dottedVer);
  if (!dry) {
    data.version = dottedVer;
    data.version_name = requestedVersion;
    fs.writeFileSync(p, JSON.stringify(data, undefined, 2) + "\n");
  }
}

let debs = ["taler-wallet-cli", "taler-harness"];

for (const deb of debs) {
  const p = `packages/${deb}/debian/changelog`;
  const data = fs.readFileSync(p, {
    encoding: "utf-8",
  });
  const lines = data.split("\n");
  for (const line of lines) {
    const s = line.trim();
    if (s == "") {
      continue;
    }
    const re = /\((.*)\)/;
    const m = s.match(re);
    const version = m[1];
    let pfx = "";
    // In Debian, the pre-release version is indicated
    // by tilde, not by dash.
    const debVersion = requestedVersion.replace("-", "~")
    if (version != debVersion) {
      pfx = "[!] ";
      if (!dry) {
        const dateStr = child_process.execSync("date -R", {
          encoding: "utf-8",
        });
        const entryLines = [
          `${deb} (${debVersion}) unstable; urgency=low`,
          "",
          `  * Release ${requestedVersion}`,
          "",
          ` -- Florian Dold <dold@taler.net>  ${dateStr}`,
          ``,
        ];
        fs.writeFileSync(p, `${entryLines.join("\n")}${data}`);
      }
    }
    console.log(`${pfx}${p} is ${version}`);
    break;
  }
}
