/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { codecForUIForms, UiForms } from "@gnu-taler/web-util/browser";
import { ComponentChildren, createContext, h, VNode } from "preact";
import { useContext } from "preact/hooks";

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

export type Type = UiForms;

const defaultForms: UiForms = {
  forms: [],
};

const Context = createContext<Type>(defaultForms);

export type BaseForm = Record<string, unknown>;

export const useUiFormsContext = (): Type => useContext(Context);

export const UiFormsProvider = ({
  children,
  value,
}: {
  value: UiForms;
  children: ComponentChildren;
}): VNode => {
  return h(Context.Provider, {
    value,
    children,
  });
};

function removeUndefineField<T extends object>(obj: T): T {
  const keys = Object.keys(obj) as Array<keyof T>;
  return keys.reduce((prev, cur) => {
    if (typeof prev[cur] === "undefined") {
      delete prev[cur];
    }
    return prev;
  }, obj);
}

export function fetchUiForms(listener: (s: UiForms) => void): void {
  fetch("./forms.json")
    .then((resp) => resp.json())
    .then((json) => codecForUIForms().decode(json))
    .then((result) =>
      listener({
        ...defaultForms,
        ...removeUndefineField(result),
      }),
    )
    .catch((e) => {
      console.log("failed to fetch forms", e);
      listener(defaultForms);
    });
  return;
}
