/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { MerchantApiClient, PreparePayResultType } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV3,
} from "../harness/helpers.js";
import { SyncService } from "../harness/sync.js";

export async function runWalletBackupDoublespendTest(t: GlobalTestState) {
  // Set up test environment

  const { commonDb, merchant, walletClient, bankClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  const sync = await SyncService.create(t, {
    currency: "TESTKUDOS",
    annualFee: "TESTKUDOS:0.5",
    database: commonDb.connStr,
    fulfillmentUrl: "taler://fulfillment-success",
    httpPort: 8089,
    name: "sync1",
    paymentBackendUrl: merchant.makeInstanceBaseUrl(),
    uploadLimitMb: 10,
  });

  await sync.start();
  await sync.pingUntilAvailable();

  await walletClient.call(WalletApiOperation.AddBackupProvider, {
    backupProviderBaseUrl: sync.baseUrl,
    activate: true,
    name: sync.baseUrl,
  });

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:10",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  await walletClient.call(WalletApiOperation.RunBackupCycle, {});
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
  await walletClient.call(WalletApiOperation.RunBackupCycle, {});

  const backupRecovery = await walletClient.call(
    WalletApiOperation.ExportBackupRecovery,
    {},
  );

  const { walletClient: walletClientTwo } = await createWalletDaemonWithClient(
    t,
    { name: "default" },
  );

  await walletClientTwo.call(WalletApiOperation.ImportBackupRecovery, {
    recovery: backupRecovery,
  });

  await walletClientTwo.call(WalletApiOperation.RunBackupCycle, {});

  console.log(
    "wallet1 balance before spend:",
    await walletClient.call(WalletApiOperation.GetBalances, {}),
  );

  await makeTestPaymentV2(t, {
    merchant,
    walletClient,
    order: {
      summary: "foo",
      amount: "TESTKUDOS:7",
    },
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  console.log(
    "wallet1 balance after spend:",
    await walletClient.call(WalletApiOperation.GetBalances, {}),
  );

  {
    console.log(
      "wallet2 balance:",
      await walletClientTwo.call(WalletApiOperation.GetBalances, {}),
    );
  }

  // Now we double-spend with the second wallet

  {
    const instance = "default";

    const orderResp = await merchantClient.createOrder({
      order: {
        amount: "TESTKUDOS:8",
        summary: "bla",
        fulfillment_url: "taler://fulfillment-success",
      },
    });

    let orderStatus = await merchantClient.queryPrivateOrderStatus({
      orderId: orderResp.order_id,
    });

    t.assertTrue(orderStatus.order_status === "unpaid");

    // Make wallet pay for the order

    {
      console.log(
        "wallet2 balance before preparePay:",
        await walletClientTwo.call(WalletApiOperation.GetBalances, {}),
      );
    }

    const preparePayResult = await walletClientTwo.call(
      WalletApiOperation.PreparePayForUri,
      {
        talerPayUri: orderStatus.taler_pay_uri,
      },
    );

    t.assertDeepEqual(
      preparePayResult.status,
      PreparePayResultType.PaymentPossible,
    );

    const res = await walletClientTwo.call(WalletApiOperation.ConfirmPay, {
      transactionId: preparePayResult.transactionId,
    });

    console.log(res);

    // FIXME: wait for a notification that indicates insufficient funds!

    await withdrawViaBankV3(t, {
      walletClient: walletClientTwo,
      bankClient,
      exchange,
      amount: "TESTKUDOS:50",
    });

    const bal = await walletClientTwo.call(WalletApiOperation.GetBalances, {});
    console.log("bal", bal);

    await walletClientTwo.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
  }
}

runWalletBackupDoublespendTest.suites = ["wallet", "wallet-backup"];
// See https://bugs.taler.net/n/7598
runWalletBackupDoublespendTest.experimental = true;
