/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { PAGINATED_LIST_REQUEST } from "../utils/constants.js";

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import {
  AccessToken,
  TalerHttpError,
  TalerMerchantManagementResultByMethod,
} from "@gnu-taler/taler-util";
import _useSWR, { SWRHook, mutate } from "swr";
import { useSessionContext } from "../context/session.js";
import { buildPaginatedResult } from "@gnu-taler/web-util/browser";
const useSWR = _useSWR as unknown as SWRHook;

export interface InstanceIncomingTransferFilter {
  payto_uri?: string;
  verified?: boolean;
  confirmed?: boolean;
  position?: string;
}
export interface InstanceConfirmedTransferFilter {
  payto_uri?: string;
  expected?: boolean;
  position?: string;
}

export function revalidateInstanceIncomingTransfers() {
  return mutate(
    (key) => Array.isArray(key) && key[key.length - 1] === "listIncomingWireTransfers",
    undefined,
    { revalidate: true },
  );
}
export function useInstanceIncomingTransfers(
  args?: InstanceIncomingTransferFilter,
  updatePosition: (id: string | undefined) => void = () => {},
) {
  const { state, lib } = useSessionContext();

  async function fetcher([token, o, p, v, c]: [
    AccessToken,
    string,
    string,
    boolean,
    boolean,
  ]) {
    return await lib.instance.listIncomingWireTransfers(token, {
      paytoURI: p,
      verified: v,
      confirmed: c,
      limit: PAGINATED_LIST_REQUEST,
      offset: o,
      order: "dec",
    });
  }

  const { data, error } = useSWR<
    TalerMerchantManagementResultByMethod<"listIncomingWireTransfers">,
    TalerHttpError
  >(
    [
      state.token,
      args?.position,
      args?.payto_uri,
      args?.verified,
      args?.confirmed,
      "listIncomingWireTransfers",
    ],
    fetcher,
  );

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  return buildPaginatedResult(
    data.body.transfers,
    args?.position,
    updatePosition,
    (d) => String(d.transfer_serial_id),
    PAGINATED_LIST_REQUEST,
  );
}
export function revalidateInstanceConfirmedTransfers() {
  return mutate(
    (key) => Array.isArray(key) && key[key.length - 1] === "listConfirmedWireTransfers",
    undefined,
    { revalidate: true },
  );
}
export function useInstanceConfirmedTransfers(
  args?: InstanceConfirmedTransferFilter,
  updatePosition: (id: string | undefined) => void = () => {},
) {
  const { state, lib } = useSessionContext();

  async function fetcher([token, o, p, e]: [
    AccessToken,
    string,
    string,
    boolean,
  ]) {
    return await lib.instance.listConfirmedWireTransfers(token, {
      paytoURI: p,
      expected: e,
      limit: PAGINATED_LIST_REQUEST,
      offset: o,
      order: "dec",
    });
  }

  const { data, error } = useSWR<
    TalerMerchantManagementResultByMethod<"listConfirmedWireTransfers">,
    TalerHttpError
  >(
    [
      state.token,
      args?.position,
      args?.payto_uri,
      args?.expected,
      "listConfirmedWireTransfers",
    ],
    fetcher,
  );

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  return buildPaginatedResult(
    data.body.transfers,
    args?.position,
    updatePosition,
    (d) => String(d.transfer_serial_id),
    PAGINATED_LIST_REQUEST,
  );
}
