/*
 This file is part of GNU Taler
 (C) 2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AbsoluteTime,
  Duration,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  GlobalTestState,
} from "../harness/harness.js";

const coinCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  feeDeposit: "TESTKUDOS:0",
  feeRefresh: "TESTKUDOS:0",
  feeRefund: "TESTKUDOS:0",
  feeWithdraw: "TESTKUDOS:0",
  rsaKeySize: 1024,
};

const coinConfigList: CoinConfig[] = [
  {
    ...coinCommon,
    name: "n1",
    value: "TESTKUDOS:1",
  },
];

const purse_expiration = AbsoluteTime.toProtocolTimestamp(
  AbsoluteTime.addDuration(
    AbsoluteTime.now(),
    Duration.fromSpec({ days: 2 }),
  ),
);

/**
 * Test peer pull payments with a large number of coins.
 *
 * Since we use an artificallly large number of coins, this
 * test is a bit slower than other tests.
 */
export async function runPeerPullLargeTest(t: GlobalTestState) {
  // Set up test environment
  const [
    { walletClient: wallet1, bankClient, exchange },
    { walletClient: wallet2 },
  ] = await Promise.all([
    createSimpleTestkudosEnvironmentV3(t, coinConfigList),
    createWalletDaemonWithClient(t, {
      name: "w2"
    })
  ]);

  // Withdraw digital cash into the wallet.
  const withdrawRes = await withdrawViaBankV3(t, {
    walletClient: wallet2,
    bankClient,
    exchange,
    amount: "TESTKUDOS:500",
  });
  await withdrawRes.withdrawalFinishedCond;

  const initiate = await wallet1.call(
    WalletApiOperation.InitiatePeerPullCredit,
    {
      exchangeBaseUrl: exchange.baseUrl,
      partialContractTerms: {
        summary: "Hello World",
        amount: "TESTKUDOS:200",
        purse_expiration,
      },
    },
  );

  await wallet1.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: initiate.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.Ready
    },
  });

  const tx = await wallet1.call(WalletApiOperation.GetTransactionById, {
    transactionId: initiate.transactionId,
  });

  t.assertDeepEqual(tx.type, TransactionType.PeerPullCredit);
  t.assertTrue(!!tx.talerUri);

  const prepare = await wallet2.client.call(
    WalletApiOperation.PreparePeerPullDebit,
    {
      talerUri: tx.talerUri,
    },
  );

  await wallet2.client.call(WalletApiOperation.ConfirmPeerPullDebit, {
    transactionId: prepare.transactionId,
  });

  await Promise.all([
    wallet1.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: initiate.transactionId,
      txState: {
        major: TransactionMajorState.Done,
      },
    }),
    wallet2.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: prepare.transactionId,
      txState: {
        major: TransactionMajorState.Done,
      },
    })
  ]);
}

runPeerPullLargeTest.suites = ["wallet", "slow"];
