/**
 * @brief  Taglist functions
 *
 * This file contains the functions to handle taglists.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    libpbbipc/pbb_taglist.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 *
 * @page taglist Taglists - PBButtonsds basic data exchange concept
 *
 * Each of pbbuttonsd modules need parameters from outside to work
 * efficiently and many of them have information other modules are
 * interested in. Furthermore pbbuttonsd need to exchange data with
 * its clients. So a flexible data transportation layer is needed,
 * where its users must not know the receivers identity.
 *
 * In pbbuttonsd this task is solved by so called <b>tags</b>. A
 * tagitem is a small structure with following contents. It is defined
 * in @ref pbbtaglist.h.
 *
 * <pre>
 *   struct tagitem {
 *       long    tag;     Identifier
 *       long    data;    Argument
 *   }
 * </pre>
 *
 * The tag is the identifier of the attached data. So the parameter,
 * its data type and meaning are defined through the tag, which on
 * the other side identifies the parameter.
 * 
 * If multiple tag-items were queued in an array, we would get a
 * tag-list. This tag-list could hold any count of parameters. The
 * end of a tag-list is markes with TAG_END and is defined  in
 * <i>taglist.h</i>.
 *
 * Any data type which is short enough to fit into a long, could be
 * stored into a tag-item. In any other case tag-data could contain a
 * pointer to the data. But be careful, because of the strict borders
 * between processes those pointers couldn't be sent to clients, which
 * are usually other processes with a own address space. PBButtonsd IPC
 * module takes care of strings and transports them correctly to and
 * from clients, but all other pointers will be invalid on client side.
 *
 * The libpbbipc contains a whole bunch of functions to cope with
 * taglists. The interface is defined in @ref pbb_taglist.c
 */

#include "pbbtaglist.h"

/**
 * @brief  Find a tag and returns its value
 *
 * This function searches for a given 'tag' in 'taglist'. If
 * the tag was found its data otherwise 'val' would be returned.
 *
 * @param  *taglist  Array of tagitems to search in
 * @param  tag       tag value to look for
 * @param  val       default value. This value will be returned, if the
 *                   tag can't be found in taglist.
 * @return the data of the tag or the default value 'val'
 */
tag_t
tagfind (struct tagitem *taglist, tag_t tag, tag_t val)
{
	while (taglist->tag != TAG_END) {
		if (taglist->tag == tag) {
			val = taglist->data;
			break;
		} else
			taglist++;
	}
	return val;
}

/**
 * @brief  Mark a tag as invalid
 *
 * This function looks for a tag in taglist and mark it as invalid.
 * In fact the tag value will be replaced by TAG_SKIP. As result 
 * the original tag is removed from the taglist and can't be used
 * anymore.
 *
 * If taglist doesn't contain Tag, nothing will be changed.
 *
 * @param  *taglist  Array of tagitems to search in
 * @param  tag       tag value to mark as invalid
 */
void
tagskip (struct tagitem *taglist, tag_t tag)
{
	while (taglist->tag != TAG_END) {
		if (taglist->tag == tag) {
			taglist->data = taglist->tag;
			taglist->tag = TAG_SKIP;
			break;
		} else
			taglist++;
	}
}

/**
 * @brief  Mark a tagitem as error and store an error code
 *
 * If an module can't use a tagitem because its data is invalid, the
 * module can use this function to return an error code to the sender
 * of the tagitem.
 *
 * The tag will be marked as error and the data field will be replaced
 * with an error code.
 *
 * @param  *tag   Tagitem to store the error in
 * @param  error  error code
 * @return a copy of error
 */
int
tagerror (struct tagitem *tag, int error)
{
	tag->data = (tag_t) error;
	tag->tag |= FLG_ERROR;
	return error;
}

/* This function initializes a taglist. */

void
taglist_init (struct tagitem *taglist)
{
	taglist->tag = TAG_END;
	taglist->data = 0;
}

/* This function adds a tagitem to an existing taglist. The taglist must be
   long enough for the new tag. No boundary checks are performed. */

void
taglist_add (struct tagitem *taglist, tag_t tag, tag_t val)
{
	while (taglist->tag != TAG_END)
		taglist++;     /* find end of taglist */

	taglist->tag = tag;     /* add tagitem */
	taglist->data = val;
	taglist++;
	taglist->tag = TAG_END;   /* mark end of taglist again */
	taglist->data = 0;
}

struct tagitem*
find_tagitem (struct tagitem *taglist, tag_t tag)
{
	while (taglist->tag != TAG_END && taglist->tag != tag)
		taglist++;
	if (taglist->tag == TAG_END)
		return 0;
	else
		return taglist;
}

int
taglist_count (struct tagitem *taglist)
{
	int n = 0;
	while (taglist->tag != TAG_END) {
		n++; taglist++;
	}
	return n;
}
