/*
** Copyright (C) 2001-2010 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/

#include <string.h>
#include <assert.h>

#include "buffer.h"

struct _Buffer {
	char*  	  _data;
	unsigned int    _len;
};

Buffer*
buffer_new (void)
{
	Buffer *new_buffer;

	new_buffer = (Buffer*)malloc (sizeof(Buffer));
	if (!new_buffer)
		return NULL;

	new_buffer->_data  = NULL;
	new_buffer->_len   = 0;

	return new_buffer;
}


void 
buffer_destroy (Buffer *buffer)
{
	assert (buffer);
	
	buffer->_len = 0;
	free (buffer->_data);
	buffer->_data = NULL;

	free (buffer);
}


int
buffer_manage_data (Buffer* buffer, char* data, unsigned int len)
{
	assert (buffer);
	assert (data || len == 0);

	free (buffer->_data);

	buffer->_data = data;
	buffer->_len  = len;

	return len;
}



char        
buffer_at (const Buffer* buffer, BufferIterator pos)
{       
	assert (buffer);
	assert (buffer->_data);
	/* assert (pos >= buffer_begin(pos)); */
	assert (pos < buffer_end(buffer));
	
	return buffer->_data[pos];
}
                   

	
const char* 
buffer_data (const Buffer* buffer)
{
	assert (buffer);

	return buffer->_data;
}

char *
buffer_data_to_string (const Buffer* buffer)
{
	char *data;

	assert (buffer);

	data = (char*)malloc(buffer->_len + 1);
	if (!data)
		return NULL;

	memcpy (data, buffer->_data, buffer->_len);
	data[buffer->_len] = '\0';

	return data;
}



const char* 
buffer_data_pos (const Buffer* buffer, BufferIterator pos)
{
	assert (buffer);
	assert (buffer->_data);
	assert (pos < buffer_length(buffer));
	/* assert (pos >= buffer_begin(pos)); */
	assert (pos < buffer_end(buffer));


	return buffer->_data + pos;
}

unsigned int 
buffer_length (const Buffer *buffer)
{
	assert (buffer);

	return buffer->_len;
}

	
int
buffer_append (Buffer* buffer, const char* data, unsigned int len)
{
	char *newdata;

	assert (buffer);
	assert (data || len == 0);

	newdata = (char*)realloc(buffer->_data, buffer->_len + len);
	if (!newdata)
		return -1;
		
	memcpy (newdata + buffer->_len, data, len);
	
	buffer->_data = newdata;
	buffer->_len += len;

	return buffer->_len;
}


unsigned int
buffer_erase (Buffer *buffer, unsigned int pos, unsigned int len)
{
	assert (buffer);
	assert (buffer->_len >= pos + len);
	
	memmove (buffer->_data + pos,
		 buffer->_data + pos + len,
		 buffer->_len  - (pos + len));
	
	buffer->_len  -= len;
	
	return buffer->_len;
}




BufferIterator 
buffer_find_offset (const Buffer *buffer, unsigned int offset, const char* pattern, 
		    unsigned int len)
{	
	unsigned int i, j;
	int match = 1;
	
	assert (buffer);
	/* assert (buffer->_len >= 0); */
	assert (pattern || len == 0);
	assert (offset < buffer->_len);
	
	for (i = offset; i < buffer->_len - len; ++i) {
		for (j = 0; j < len; ++j) {
			match = (buffer->_data[i+j] == pattern[j]);
			if (!match) 
				break;
		}
		if (match && (j == len)) 
			break;
	}
	
	if ( i == buffer->_len - len)
		i = buffer_end(buffer);
	
	return i;
}


void
buffer_clear (Buffer *buffer)
{
	assert (buffer);

	free (buffer->_data);

	buffer->_data = NULL;
	buffer->_len  = 0;
}
