/* cmd-my-id.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "tla/libarch/my.h"
#include "tla/libarch/namespace.h"
#include "tla/libarch/cmd-rbrowse.h"
#include "tla/libarch/patch-logs.h"
#include "tla/libarch/project-tree.h"
#include "tla/libarch/archive.h"
#include "tla/libarch/archives.h"




static t_uchar * usage = "[options] [REGULAR_EXPRESSION]";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
                                   "\n"
                                   "Copyright 2003 Tom Lord\n"
                                   "\n"
                                   "This is free software; see the source for copying conditions.\n"
                                   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
                                   "PARTICULAR PURPOSE.\n"
                                   "\n"
                                   "Report bugs to " cfg__tla_bug_mail ".\n"
                                   "\n"
                                   cfg__std__release_id_string
                                   "\n");

#define OPTS(OP, OP2) \
OP (opt_help_msg, "h", "help", 0, \
    "Display a help message and exit.") \
OP (opt_long_help, "H", 0, 0, \
    "Display a verbose help message and exit.") \
OP (opt_version, "V", "version", 0, \
    "Display a release identifier string") \
OP (opt_all_archives, "a", "search-all", 0, \
    "Search all archives") \
OP (opt_archive, "A", "archive", 1, \
    "Use [archive] instead of default") 

t_uchar arch_cmd_rbrowse_help[] = ("print an outline describing an archive's contents\n"
                                   "\n"
                                   "an outline of an archive will be printed, showing the\n"
                                   "categories, branches and revisions of the specified archive\n"
                                   "if no archive is given then `my-default-archive` is used" );

enum options

{
  OPTS (OPT_ENUM, OPT_IGN)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_rbrowse (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  int all_archives = 0;
  struct opt_parsed * option;
  char * errname;
  t_uchar * default_archive = NULL;
  t_uchar * regex = NULL;
  regex_t regex_needle;
  rel_table archive_list = NULL;
  int w;


  errname = argv[0];

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, arch_cmd_rbrowse_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_all_archives:
            {
              all_archives = 1;
              break;
            }
        case opt_archive:
            {
              default_archive = str_save(0, option->arg_string);
              break;
            }

        }
    }

  if (argc > 2)
    goto usage_error;

  if (argc == 2)
    regex = str_save(0, argv[1]);
  else
    regex = str_save(0, ".*");

  if (regcomp(&regex_needle, regex, REG_EXTENDED))
    panic ("Invalid regular expression given");


  default_archive = arch_my_default_archive (default_archive);

  if (!arch_valid_archive_name (default_archive))
    panic ("Invalid archive given");

  if (! all_archives)
    {
      rel_add_records (&archive_list, rel_make_record (default_archive, 0), 0);
    }
  else
    {
      archive_list = arch_registered_archives ();
    }

  for (w = 0; w < rel_n_records (archive_list); w++)
    {
      int x;
      int y;
      int z;
      int shown_archive = 0;
      struct arch_archive * arch = 0;
      rel_table categories;

      if (str_cmp("-MIRROR", str_chr_rindex(archive_list[w][0], '-')) == 0 ||
          str_cmp("-SOURCE", str_chr_rindex(archive_list[w][0], '-')) == 0 )
        continue;

      arch = arch_archive_connect(archive_list[w][0], 0);

      categories = arch_archive_categories(arch);

      for (x = 0; x < rel_n_records (categories); ++x)
        {


          int shown_category = 0;
          int category_needshow = 0;
          rel_table branches = 0;

          if (regexec (&regex_needle, categories[x][0], 0, 0, 0) == 0)
            category_needshow = 1;

          branches = arch_archive_branches(arch, categories[x][0]);

          for (y = 0; y < rel_n_records (branches); ++y)
            {
              int shown_branch = 0;
              int branch_needshow = 0;
              rel_table versions = 0;

              if (regexec (&regex_needle, branches[y][0], 0, 0, 0) == 0)
                branch_needshow = 1;

              versions = arch_archive_versions(arch, branches[y][0]);

              for (z = 0; z < rel_n_records (versions); ++z)
                {
                  int needshow = 0;
                  int version_needshow = 0;

                  if (regexec (&regex_needle, versions[z][0], 0, 0, 0) == 0)
                    version_needshow = 1;

                  needshow = (version_needshow || branch_needshow || category_needshow);

                  if (needshow)
                    {
                      if (!shown_archive)
                        {
                          shown_archive = 1;
                          safe_printfmt (1,"%s\n", archive_list[w][0]);
                        }
                      if (!shown_category)
                        {
                          shown_category = 1;
                          safe_printfmt (1, "   %s\n", categories[x][0]);
                        }
                      if (!shown_branch)
                        {
                          shown_branch = 1;
                          safe_printfmt(1, "      %s\n", branches[y][0]);
                        }

                      safe_printfmt(1, "         %s\n\n", versions[z][0]);
                    }
                }
            }
        }
    }

  return 0;
}




/* tag: James Blackwell Thu Dec 11 16:50:14 EST 2003 (cmd-rbrowse.c)
*/
