#@ MODIF N_SENSIBILITE Noyau  DATE 07/09/2009   AUTEUR COURTOIS M.COURTOIS 
# -*- coding: iso-8859-1 -*-
# RESPONSABLE COURTOIS M.COURTOIS
#            CONFIGURATION MANAGEMENT OF EDF VERSION
# ======================================================================
# COPYRIGHT (C) 1991 - 2006  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY  
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY  
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR     
# (AT YOUR OPTION) ANY LATER VERSION.                                                  
#                                                                       
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT   
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF            
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU      
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.                              
#                                                                       
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE     
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,         
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.        
# ======================================================================

"""
   Ce module contient les rgles ncessaires aux commandes sensibles
   pour renseigner l'attribut etape.sd.sensi, grer le caractre rentrant
   sur prsence de la sensibilit.
"""

from types import TupleType, ListType
EnumTypes = (TupleType, ListType)

from N_REGLE import REGLE

# -----------------------------------------------------------------------------
class CONCEPT_SENSIBLE(REGLE):
   """Rgle permettant de renseigner au niveau du catalogue comment sera
   rempli le concept (valeur nominale ou drive(s) ou les deux...).
   """
   def __init__(self, mode, mocle='SENSIBILITE'):
      """Constructeur.

         mode : manire dont la commande rempli le concept
            - 'ENSEMBLE' : concept nominal ET drives en une seule passe
            - 'SEPARE'   : concept nominal OU drive (une ou plusieurs)
            
         mocle : mot-cl contenant les paramtres sensibles.
      """
      REGLE.__init__(self)
      self.mocle = mocle
      self._modes = { 'ENSEMBLE' : 0, 'SEPARE' : 1 }
      self.mode = self._modes.get(mode, self._modes['ENSEMBLE'])

   def gettext(self):
      """Pour EFICAS
      """
      return ''

   def verif(self, args):
      """Retourne texte + 1 si ok, 0 si nook.
      On stocke dans sd.sensi l'tape courante, c'est--dire celle qui
      renseigne le concept si cela n'a pas dj t fait (car verif est
      appel  chaque validation).
      """
      obj = args["self"]
      etape = obj.etape
      id_etape = '%s_%s' % (etape.id, id(etape))
      if etape.sd == None:
          return '',1
      if not hasattr(etape.sd,"sensi"):
         etape.sd.sensi = {}
      # si ENSEMBLE, la sd nominale est forcment produite
      if self.mode == self._modes['ENSEMBLE'] and not etape.sd.sensi.has_key('nominal'):
         etape.sd.sensi['nominal'] = id_etape
      # liste des paramtres sensibles
      valeur = obj[self.mocle]
      if valeur == None:
         # pas de sensibilit, la sd nominale est produite
         if not etape.sd.sensi.has_key('nominal'):
            etape.sd.sensi['nominal'] = id_etape
         return '', 1
      if not type(valeur) in EnumTypes:
         valeur = [valeur,]
      for v in valeur:
         if not etape.sd.sensi.has_key(v.get_name()):
            etape.sd.sensi[v.get_name()] = id_etape
      return '', 1


# -----------------------------------------------------------------------------
class REUSE_SENSIBLE(REGLE):
   """Limite le caractre rentrant de la commande.
   On autorisera reuse seulement si le concept (au sens fortran) n'a pas dj
   t calcul (d'aprs sd.sensi). Ce sera interdit dans les cas suivants :
      - sd nominale calcule et SENSIBILITE absent
      - PS1 dans SENSIBILITE et sd drive par rapport  PS1 calcule
   """
   def __init__(self, mocle='SENSIBILITE'):
      """Constructeur.
         mocle : mot-cl SENSIBILITE.
      """
      REGLE.__init__(self)
      self.mocle = mocle

   def gettext(self):
      """Pour EFICAS
      """
      return ''

   def verif(self,args):
      """Retourne texte + 1 si ok, 0 si nook = reuse interdit.
      Comme CONCEPT_SENSIBLE est appel avant (et  chaque validation),
      on regarde si sd.sensi[ps] a t renseign par une tape prcdente.
      """
      obj = args["self"]
      etape = obj.etape
      id_etape = '%s_%s' % (etape.id, id(etape))
      sd = etape.sd
      # si la commande n'est pas rentrante, rien  faire
      if etape.reuse is not None:
         valeur = obj[self.mocle]
         if valeur is None:
            if not hasattr(sd, 'sensi') or sd.sensi.get('nominal', id_etape) != id_etape:
               # pas de sensibilite et concept nominal dj calcul : reuse interdit
               text = "Commande non rentrante en l'absence de sensibilit."
               return text, 0
         else:
            if not type(valeur) in EnumTypes:
               valeur = [valeur,]
            for ps in valeur:
               if hasattr(sd, 'sensi') and sd.sensi.get(ps.nom, id_etape) != id_etape:
                  # concept driv par rapport  ps dj calcul : reuse interdit
                  text = "Commande non rentrante : drive par rapport  %s dj calcule" % ps.nom
                  return text, 0
      return '', 1


# -----------------------------------------------------------------------------
class DERIVABLE(REGLE):
   """Dclare que le concept fourni derrire un mot-cl est drivable.
   Sa prsence ne suffit pas  le valider, il faut encore que son attribut
   '.sensi' soit cohrent avec le contenu du mot-cl SENSIBILITE (ou l'absence
   de celui-ci).
   """
   def __init__(self, mocle):
      """Constructeur.
         mocle : mot-cl drivable.
      """
      REGLE.__init__(self)
      self.mocle = mocle

   def gettext(self):
      """Pour EFICAS
      """
      return ''

   def verif(self,args):
      """
      """
      obj = args["self"]
      try:
         concept = obj[self.mocle]
      except IndexError:
         return '', 1
      if not type(concept) in EnumTypes:
         concept = [concept,]
      l_ps = obj["SENSIBILITE"]
      for co in concept:
         if co is None:
            text = "Concept non dfini (None) sous le mot-cl %s" % self.mocle
            return text, 0
         if not l_ps:
            # pas de sensibilit
            if hasattr(co,"sensi") and not co.sensi.get('nominal'):
               text = "%s ne contient que des valeurs drives, utilisez le mot cle SENSIBILITE" %\
                     co.nom
               return text, 0
         else:
            # sensibilit spcifie
            if not type(l_ps) in EnumTypes:
               l_ps = [l_ps,]
            for ps in l_ps:
               if not hasattr(co,"sensi") or not co.sensi.get(ps.nom):
                  text = "La drive de %s par rapport  %s n'est pas disponible." %\
                        (co.nom, ps.nom)
                  return text, 0
      return '', 1

