/*

  Virtual Observatory capable cone search

  Copyright © 2010 - 2014, 2018-9 F.Hroch (hroch@physics.muni.cz)

  This file is part of Munipack.

  Munipack is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Munipack is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "votable.h"
#include "vocatconf.h"
#include "voclient.h"
#include <wx/wx.h>
#include <wx/app.h>
#include <wx/url.h>
#include <wx/wfstream.h>
#include <wx/txtstrm.h>
#include <wx/filename.h>
#include <wx/filefn.h>
#include <fitsio.h>
#include <list>
#include <cstdio>
#include <cmath>


using namespace std;

void PatchJohn(const wxString&);

class Cone: public wxAppConsole
{
public:
  bool OnInit();
  int OnRun();
private:
  wxString conesearch();
  wxString tmpcone;
};
IMPLEMENT_APP_CONSOLE(Cone)


bool Cone::OnInit()
{
  wxLog::DisableTimestamp();
  tmpcone = wxFileName::CreateTempFileName("votcone_");
  return true;
}

int Cone::OnRun()
{
  wxString msg = conesearch();

  // remove cache file
  if( msg == "" )
    wxRemoveFile(tmpcone);

  // It is just strangle workaround.
  // The code simulates STOP commands which terminate of Fortran programs.
  // We have no faith to exit codes returned by child processes because
  // a correct terminate would leads to generate 'Child process (PID XXX)
  // still alive but pipe closed so generating a close notification'
  // under some GNU/Linux distributions (Fedora). Processing of STOP
  // eliminates use of process execution terminate codes.
  // One looks as a feature (bug) of WX.
  if( msg == "" )
    fprintf(stderr,"STOP 0\n");
  else
    fprintf(stderr,"STOP '%s'\n",static_cast<const char *>(msg.c_str()));

  return msg != "";
}

wxString Cone::conesearch()
{
  const wxString amp("&");
  wxString url, catname, output, type, sort, ra, dec, sr, magmin, magmax;
  list<wxString> pars;
  bool verbose = false, pipelog = false, patch = false;


  // limit logging
  wxLog::SetLogLevel(wxLOG_Message);
  wxLog::SetVerbose(); // important to activate previous setup

  wxFFileInputStream istream(stdin);
  wxTextInputStream input(istream);

  // replace by using of regex? fortran-fread ?

  while(istream.IsOk() && ! istream.Eof()) {

    wxString line = input.ReadLine();

    if( line.StartsWith("VERBOSE") ) {
      verbose = GetBool(line);
      if( verbose )
	wxLog::SetLogLevel(wxLOG_Debug);
      else
	wxLog::SetLogLevel(wxLOG_Message);
      wxLog::SetVerbose();
    }

    if( line.StartsWith("PIPELOG") )
      pipelog = GetBool(line);

    if( line.StartsWith("PATCH") )
      patch = GetBool(line);

    if( line.StartsWith("URL") )
      url = GetString(line);

    if( line.StartsWith("CATNAME") )
      catname = GetString(line);

    if( line.StartsWith("OUTPUT") )
      output = GetString(line);

    if( line.StartsWith("TYPE") )
      type = GetString(line);

    if( line.StartsWith("SORT") )
      sort = GetString(line);

    if( line.StartsWith("MAGMIN") )
      magmin = GetString(line);

    if( line.StartsWith("MAGMAX") )
      magmax = GetString(line);

    if( line.StartsWith("PAR") )
      pars.push_back(GetString(line));
  }

  wxASSERT(!url.IsEmpty());

  if( magmin != "" && magmax == "" )
    url += amp + sort + "=>" + magmin;
  else if ( magmax != "" && magmin == "" )
    url += amp + sort + "=<" + magmax;
  else if( magmin != "" && magmax != "" )
    url += amp + sort + "=" + magmin + ".." + magmax;

  for(list<wxString>::const_iterator a = pars.begin(); a != pars.end(); ++a)
    url += (a != pars.end() ? amp : "") + *a;

  VOTable vt;

  for(size_t iter = 0; iter < 7; iter++) {

    wxURL u(url);
    if( ! u.IsOk() ) {
      if( u.GetError() == wxURL_SNTXERR )
	wxLogError("Syntax error in the URL string.");
      else if( u.GetError() == wxURL_NOPROTO )
	wxLogError("Found no protocol which can get this URL.");
      else if( u.GetError() == wxURL_NOHOST )
	wxLogError("A host name is required for this protocol.");
      else if( u.GetError() == wxURL_NOPATH )
	wxLogError("A path is required for this protocol.");
      return "URL string error";
    }

    wxString server = u.GetServer();
    wxString path = u.GetPath() + wxString("?") + u.GetQuery();

    VOclient voc;
    if( ! voc.Connect(server) )
      return "Failed to connect server "+server+".";

    if( ! voc.Get(path,tmpcone) )
      return "Connection failed.";

    vt.Load(tmpcone);
    wxLogInfo("XML IsOK? %s", vt.IsOk() ? "T" : "F");
    if( vt.IsOk() )
      break;

    wxLogWarning("Data parse failed ("+tmpcone+"), trying next round...");
    wxSleep(7);

    /*
      This strangle piece of code eliminates black magic powers.
      The black magic power sometimes randomly and unexpectedly stops
      a http data download in a random byte position which leaves
      XML tags opened, the result are being unparseable.

      The second (and more) attempts without any change in inputs
      should be more successful. I've no suspicion on the origin
      of the magic unnatural hand, which brings our data-streams:
      perhaps wxWidgets secrets, perhaps caching, ... perhaps a black cat.

    */
  }

  if( ! vt.IsOk() ) {
    if( pipelog ) wxPrintf("=CONE> Failed to open: `"+url+"'\n");
    return "Cone search failed.";
  }

  if( vt.HasError() )
    return vt.GetErrorMsg();

  if( vt.IsEmpty() )
    return "Empty table (parameters needs to by adjusted more carefully.";
  else {
    if( pipelog ) wxPrintf("=CONE> %d objects found\n",vt.RecordCount());
    wxLogInfo("Cone search: %d objects found.",vt.RecordCount());
  }

  if( ! sort.IsEmpty() )
    vt.Sort(sort);

  if( type.IsEmpty() )
    type = GetFileType(output);

  if( type == "FITS" ) {
    FITStable ft(vt);
    bool ok = ft.Save(output,true);
    if( ok ) {
      if( catname == "UCAC4" && patch )
	PatchJohn(output);
      return "";
    }
    else
      return "FITS save failed.";
  }
  else
    return vt.Save(output) ? "" : "Save failed.";

  return "";
}

void PatchJohn(const wxString& fitsname)
{
  /*
     Adds two new colums: Johnson RI filters to the table.
     Also removes the older Gunn ri columns.

     https://gaia.esac.esa.int/documentation/GDR1/Data_processing/chap_cu5phot/sec_phot_calibr.html

      http://www.sdss.org/dr4/algorithms/sdssUBVRITransform.html
      see Lupton (2005)

  */

  wxLogInfo("Transforming Gunn's ri to Johnson RI ..");

  fitsfile *fits = 0;
  int status = 0;

  if( fits_open_table(&fits,fitsname.char_str(),READWRITE,&status) ) {
    fits_report_error(stderr, status);
    exit(1);
  }

  long nrows;
  if( fits_get_num_rows(fits,&nrows,&status) ) {
    fits_report_error(stderr, status);
    exit(1);
  }

  float *rmag = new float[nrows];
  float *imag = new float[nrows];
  float *e_rmag = new float[nrows];
  float *e_imag = new float[nrows];
  char **f_rmag = new char*[nrows];
  char **f_imag = new char*[nrows];
  for(int i = 0; i < nrows; i++) {
    f_rmag[i] = new char[7];
    f_imag[i] = new char[7];
  }

  int ncol, anynul;
  fits_get_colnum(fits,CASEINSEN,(char *)"rmag",&ncol,&status);
  fits_read_col(fits,TFLOAT,ncol,1,1,nrows,NULL,rmag,&anynul,&status);
  fits_delete_col(fits,ncol,&status);
  fits_get_colnum(fits,CASEINSEN,(char *)"e_rmag",&ncol,&status);
  fits_read_col(fits,TFLOAT,ncol,1,1,nrows,NULL,e_rmag,&anynul,&status);
  fits_delete_col(fits,ncol,&status);
  fits_get_colnum(fits,CASEINSEN,(char *)"f_rmag",&ncol,&status);
  fits_read_col(fits,TSTRING,ncol,1,1,nrows,NULL,f_rmag,&anynul,&status);
  fits_delete_col(fits,ncol,&status);
  fits_get_colnum(fits,CASEINSEN,(char *)"imag",&ncol,&status);
  fits_read_col(fits,TFLOAT,ncol,1,1,nrows,NULL,imag,&anynul,&status);
  fits_delete_col(fits,ncol,&status);
  fits_get_colnum(fits,CASEINSEN,(char *)"e_imag",&ncol,&status);
  fits_read_col(fits,TFLOAT,ncol,1,1,nrows,NULL,e_imag,&anynul,&status);
  fits_delete_col(fits,ncol,&status);
  fits_get_colnum(fits,CASEINSEN,(char *)"f_imag",&ncol,&status);
  fits_read_col(fits,TSTRING,ncol,1,1,nrows,NULL,f_imag,&anynul,&status);
  fits_delete_col(fits,ncol,&status);

  if( status != 0 ) {
    fits_report_error(stderr, status);
    exit(1);
  }

  float *Rmag = new float[nrows];
  float *Imag = new float[nrows];
  float *e_Rmag = new float[nrows];
  float *e_Imag = new float[nrows];

  for(int i = 0; i < nrows; i++) {
    if( isnormal(rmag[i]) &&  isnormal(imag[i]) ) {
      float ri = rmag[i] - imag[i];
      Rmag[i] = rmag[i] - 0.2936*ri - 0.1439;
      Imag[i] = rmag[i] - 1.2444*ri - 0.3820;
      e_Rmag[i] = e_rmag[i];
      e_Imag[i] = e_imag[i];
    }
    else {
      Rmag[i] = 99.999;
      Imag[i] = 99.999;
      e_Rmag[i] = 9.999;
      e_Imag[i] = 9.999;
    }
    /*
    wxLogInfo("%d %f %f %f %f %d %d",i,Rmag[i],Imag[i],rmag[i],imag[i],
	      isnormal(rmag[i]),isnan(imag[i]));
    */
  }

  fits_insert_col(fits,ncol,(char *)"Rmag",(char *)"1E",&status);
  fits_write_col(fits,TFLOAT,ncol,1,1,nrows,Rmag,&status);
  ncol++;
  fits_insert_col(fits,ncol,(char *)"e_Rmag",(char *)"1E",&status);
  fits_write_col(fits,TFLOAT,ncol,1,1,nrows,e_Rmag,&status);
  ncol++;
  fits_insert_col(fits,ncol,(char *)"f_Rmag",(char *)"1A",&status);
  fits_write_col(fits,TSTRING,ncol,1,1,nrows,f_rmag,&status);
  ncol++;
  fits_insert_col(fits,ncol,(char *)"Imag",(char *)"1E",&status);
  fits_write_col(fits,TFLOAT,ncol,1,1,nrows,Imag,&status);
  ncol++;
  fits_insert_col(fits,ncol,(char *)"e_Imag",(char *)"1E",&status);
  fits_write_col(fits,TFLOAT,ncol,1,1,nrows,e_Imag,&status);
  ncol++;
  fits_insert_col(fits,ncol,(char *)"f_Imag",(char *)"1A",&status);
  fits_write_col(fits,TSTRING,ncol,1,1,nrows,f_imag,&status);

  fits_write_comment(fits,"",&status);
  fits_write_comment(fits," *** NOTICE ***",&status);
  fits_write_comment(fits,
      "Rmag, Imag in Johnson system, and related columns, has been derived",
      &status);
  fits_write_comment(fits,
      "from Gunn's ri magnitudes by the transformation, Lupton (2005): ",
      &status);
  fits_write_comment(fits,
      "http://www.sdss.org/dr4/algorithms/sdssUBVRITransform.html",&status);

  fits_close_file(fits, &status);
  fits_report_error(stderr,status);

  delete[] rmag; delete[] imag; delete[] e_rmag; delete[] e_imag;
  delete[] Rmag; delete[] Imag; delete[] e_Rmag; delete[] e_Imag;
  for(int i = 0; i < nrows; i++) {
    delete[] f_rmag[i];
    delete[] f_imag[i];
  }
  delete[] f_rmag; delete[] f_imag;
}
