############################################################################
##      Copyright (C) 2005 Subredu Manuel  <diablo@iasi.roedu.net>.        #
##                                                                         #
## This program is free software; you can redistribute it and/or modify    #
## it under the terms of the GNU General Public License v2 as published by #
## the Free Software Foundation.                                           #
##                                                                         #
## This program is distributed in the hope that it will be useful,         #
## but WITHOUT ANY WARRANTY; without even the implied warranty of          #
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           #
## GNU General Public License for more details.                            #
##                                                                         #
## You should have received a copy of the GNU General Public License       #
## along with this program; if not, write to the Free Software             #
## Foundation, Inc., 59 Temple Place - Suite 330, Boston,                  #
## MA 02111-1307,USA.                                                      #
############################################################################

package RoPkg::Simba::Exclude;

use strict;
use warnings;

use Scalar::Util qw(blessed);

use RoPkg::Exceptions;
use RoPkg::DBObject;

use vars qw($table $VERSION);

$VERSION='0.1.2';

$table='Excludes';
use base qw(RoPkg::DBObject);

my $pf = {
  id        => q{-},
  MirrorID  => q{-},
  CommandID => q{-},
  ExList    => q{-},
};
 
sub new {
  my ($class, %opt) = @_;
  my $self;

  $opt{methods} = $pf;
  $self = $class->SUPER::new(%opt);
  $self->{_sql_table} = $table;

  return $self;
}

sub table {
  return $table;
}

sub Add {
  my ($self) = @_;
  
  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }
  $self->chkp(qw(CommandID MirrorID ExList));
    
  return $self->SQL_Insert();
}

sub Load {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }

  if ( defined($self->{MirrorID}) and defined($self->{CommandID}) ) {
    $self->SQL_Select(qw(MirrorID CommandID));
    $self->_populate_array();
    return;
  }

  if ( defined($self->{id}) ) {
    $self->SQL_Select(qw(id));
    $self->_populate_array();
    return;
  }

  Param::Missing->throw('(MirrorID and CommandID) or id are not defined');

  return 0;
}

sub Delete {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }
  $self->chkp(qw(id));

  return $self->SQL_Delete(qw(id));
}

sub Update {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }
  $self->chkp(qw(CommandID MirrorID ExList id));

  return $self->SQL_Update(qw(id));
}

##################################
###  Database functions - END  ###
##################################


#################################
### General functions - BEGIN ###
#################################

sub _populate_array {
  my ($self) = @_;
  my @items;

  @items = split(/,/xm, $self->{ExList});
  $self->{items} = \@items;

  return (wantarray ? @items : (scalar @items));
}

sub _rebuild_items {
  my ($self) = @_;

  $self->{ExList} = join(q{,}, @{ $self->{items} });
  
  return 1;
}

sub AddItems {
  my ($self, @v_items) = @_;
  my @items;

  if (!blessed($self) ) {
    OutsideClass->throw('Called outside class instance');
  }
  return if (!@v_items);

  if ( defined($self->{items}) ) {
    @items = @{ $self->{items} };
  }

  foreach(@v_items) {
    push @items, $_;
  }
  $self->{items} = \@items;
  return $self->_rebuild_items();
}

sub GetItems {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called outside class instance');
  }

  my @items;
  if ( defined($self->{items}) ) {
    @items = @{ $self->{items} };
    return (wantarray ? @items : $#items);
  } else {
    return (wantarray ? () : -1);
  }
}

#################################
### General functions -  END  ###
#################################

1;

__END__

=head1 NAME

RoPkg::Simba::Exclude

=head1 VERSION

0.1.2

=head1 DESCRIPTION

RoPkg::Simba::Exclude is the class used by simba to manipulate a
exclude list. It has the basic sql methods (inherited from RoPkg::DBObject).

=head1 SYNOPSIS

 !#/usr/bin/perl
 
 use RoPkg::DB;
 use RoPkg::Simba::Exclude;
 
 sub main {
   my $dbp = new RoPkg::DB();
   $dbp->Add('dbi:mysql:database=mysql;host=localhost',
             'root',
             '',
             'local');

   my $c = new RoPkg::Simba::Exclude(db => $dbp, db_method => 'db_local');
   $c->id(1);
   $c->Load();
 }
 
 main();

=head1 SUBROUTINES/METHODS

All methods (besides new) raise OutsideClass exception when
called outside class instance. Also, some methods may rise
diferent exceptions. Please read the section in which the 
method is described to find out more information about exceptions.

=head2 new()

The class constructor. At this moment, it just calls
RoPkg::DBObject->new() . Please read the RoPkg::DBObject
manual page for more information about the new() parameters.

=head2 table()

Returns the name of the exclude lists database table.

=head2 ExList()

get/set the excluded items. When set behaviour is selected, the exclude
list must be passed to the method. The exclude list B<must> be set
before adding the new exclude list to database.

=head2 AddItems(@new_items)

Add @new_items to the list of excluded items and returns
the new number of excluded items.

=head2 GetItems()

Returns the list of excluded items. In scalar context
returns the number of excluded items

The following methods are get/set methods for
all fields of a mirror.

=over 3

=item *) id

=item *) MirrorID

=item *) CommandID

=back

=head2 Add()

Adds the mirror to the database. This method is a
wrapper for RoPkg::DBObject::SQL_Insert . On success
0 is returned. On error, DBI exception is raised.

=head2 Delete()

Deletes the current exclude list from the database. Before
calling this method, you should set the B<id> of the
exclude list . If you don't set the B<id> Param::Missing 
exception is raised. On database operation success,
0 is returned. On database error, DBI exception is
raised.

=head2 Update()

Update the current exclude list object with the database. Before
calling this method, you should set the B<id> of the
exclude list . If you don't set the B<id> Param::Missing 
exception is raised. On database operation success,
0 is returned. On database error, DBI exception is
raised.

=head2 Load()

Load the exclude list from the database, into
the current object. Before calling this method
you should have set B<id> or B<MirrorID> and B<CommandID>.
If none are found, then Param::Missing is raised.
On database operation success 0 is returned. On
database error, DBI exception is raised.

=head1 DIAGNOSTICS

Unpack the source, and use 'make test' command

=head1 CONFIGURATION AND ENVIRONMENT

This module does not use any configuration files or environment
variables.

=head1 DEPENDENCIES

RoPkg::DBObject and RoPkg::Exceptions

=head1 INCOMPATIBILITIES

None known to the author

=head1 BUGS AND LIMITATIONS

None known to the author

=head1 PERL CRITIC

This module is perl critic level 2 compliant (with 1 exception)

=head1 SEE ALSO

L<RoPkg::Simba::Excludes> L<RoPkg::Exceptions> L<RoPkg::Object>

=head1 AUTHOR

Subredu Manuel <diablo@iasi.roedu.net>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2005 Subredu Manuel.  All Rights Reserved.
This module is free software; you can redistribute it 
and/or modify it under the same terms as Perl itself.
The LICENSE file contains the full text of the license.

=cut
