/*
  Author: Bosma

 MIT License

Copyright (c) 2017 Bosma

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

#pragma once

#include <chrono>
#include <thread>
#include <future>
#include <mutex>
#include <sstream>

namespace Bosma {
  class InterruptableSleep {

    using Clock = std::chrono::system_clock;

    // InterruptableSleep offers a sleep that can be interrupted by any thread.
    // It can be interrupted multiple times
    // and be interrupted before any sleep is called (the sleep will immediately complete)
    // Has same interface as condition_variables and futures, except with sleep instead of wait.
    // For a given object, sleep can be called on multiple threads safely, but is not recommended as behaviour is undefined.

  public:
    InterruptableSleep() : interrupted(false) {
    }
    InterruptableSleep(const InterruptableSleep &) = delete;
    InterruptableSleep(InterruptableSleep &&) noexcept = delete;
    ~InterruptableSleep() noexcept = default;
    InterruptableSleep& operator=(const InterruptableSleep &) noexcept = delete;
    InterruptableSleep& operator=(InterruptableSleep &&) noexcept = delete;

    void sleep_for(Clock::duration duration) {
      std::unique_lock<std::mutex> ul(m);
      cv.wait_for(ul, duration, [this] { return interrupted; });
      interrupted = false;
    }
    void sleep_until(Clock::time_point time) {
      std::unique_lock<std::mutex> ul(m);
      cv.wait_until(ul, time, [this] { return interrupted; });
      interrupted = false;
    }
    void sleep() {
      std::unique_lock<std::mutex> ul(m);
      cv.wait(ul, [this] { return interrupted; });
      interrupted = false;
    }
    void interrupt() {
      std::lock_guard<std::mutex> lg(m);
      interrupted = true;
      cv.notify_one();
    }
  private:
    bool interrupted;
    std::mutex m;
    std::condition_variable cv;
  };
}
